<?php
declare(strict_types=1);

namespace FoundationCommerce\BrandCategories\ViewModel;

use Magento\Catalog\Api\CategoryListInterface;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaBuilderFactory;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\View\Element\Block\ArgumentInterface;
use Magento\Store\Model\StoreManagerInterface;

class BrandCategoriesProvider implements ArgumentInterface
{
    /**
     * @var SearchCriteriaBuilderFactory
     */
    protected $searchCriteriaBuilderFactory;
    /**
     * @var CategoryListInterface
     */
    protected $categoryList;
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;
    /**
     * @var SortOrderBuilder
     */
    protected $sortOrderBuilder;

    public function __construct(
        CategoryListInterface $categoryList,
        SearchCriteriaBuilderFactory $searchCriteriaBuilderFactory,
        SortOrderBuilder $sortOrderBuilder,
        StoreManagerInterface $storeManager
    ) {
        $this->searchCriteriaBuilderFactory = $searchCriteriaBuilderFactory;
        $this->categoryList = $categoryList;
        $this->storeManager = $storeManager;
        $this->sortOrderBuilder = $sortOrderBuilder;
    }

    public function getAllBrandCategories(): SearchResultsInterface
    {
        $searchCriteria = $this->getBrandCategorySearchCriteriaBuilder()->create();

        return $this->categoryList->getList($searchCriteria);
    }

    public function getBrandLogoUrl(string $file): string
    {
        return $this->storeManager->getStore()->getBaseUrl() . $file;
    }

    public function getBrandLogoDataForProduct(ProductInterface $product): array
    {
        $brandData = [];
        $categoryIds = $product->getCategoryIds();
        $brandsSearchCriteriaBuilder = $this->getBrandCategorySearchCriteriaBuilder();
        $brandsSearchCriteria = $brandsSearchCriteriaBuilder
            ->addFilter('entity_id', $categoryIds, 'in')
            ->create();

        $categories = $this->categoryList->getList($brandsSearchCriteria);

        if (!$categories->getTotalCount()) {
            return $brandData;
        }

        $brand = $categories->getItems()[0];

        if (!$brand->getData('totonic_brand_logo')) {
            return $brandData;
        }

        $brandData = [
            'name' => $brand->getName(),
            'url' => $brand->getUrl(),
            'logo' => $this->getBrandLogoUrl($brand->getData('totonic_brand_logo'))
        ];

        return $brandData;
    }

    protected function getBrandCategorySearchCriteriaBuilder(): SearchCriteriaBuilder
    {
        $sortOrder = $this->sortOrderBuilder
            ->setField('position')
            ->setAscendingDirection()
            ->create();

        /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
        $searchCriteriaBuilder = $this->searchCriteriaBuilderFactory->create();

        return $searchCriteriaBuilder
            ->addFilter('is_active', 1)
            ->addFilter('totonic_is_brand', true, 'eq')
            ->addSortOrder($sortOrder);
    }
}
